<?php

namespace App\services;

use App\core\Logger;
use PDO;

class WebhookHandler
{
    private PDO $db;
    private PaymentService $paymentService;
    private Logger $logger;

    public function __construct(PDO $db, PaymentService $paymentService, Logger $logger)
    {
        $this->db = $db;
        $this->paymentService = $paymentService;
        $this->logger = $logger;
    }

    public function handle(array $payload, string $signature): array
    {
        if (!$this->paymentService->validateSignature($payload, $signature)) {
            http_response_code(401);
            return ['error' => 'invalid_signature'];
        }

        $txId = $payload['transaction_id'] ?? '';
        $amount = (float) ($payload['amount'] ?? 0);
        $referenceId = (int) ($payload['reference_id'] ?? 0);
        $referenceType = $payload['reference_type'] ?? 'recharge';

        // idempotency
        $stmt = $this->db->prepare('SELECT id FROM transactions WHERE meta->>"$.tx_id" = :tx LIMIT 1');
        $stmt->execute(['tx' => $txId]);
        if ($stmt->fetch()) {
            return ['status' => 'ok', 'idempotent' => true];
        }

        $this->logger->audit(null, 'webhook_received', $referenceType === 'order' ? 'orders' : 'recharge_requests', $referenceId, $payload);

        if ($referenceType === 'order') {
            return $this->paymentService->markOrderPaid($referenceId, $txId, $amount);
        }

        $this->paymentService->markPaid($referenceId, $txId);
        return ['status' => 'ok'];
    }
}

