<?php

namespace App\services;

use App\core\Logger;
use PDO;
use App\services\DeliveryService;
use App\services\FraudService;
use App\services\NotificationService;

class PaymentService
{
    private PDO $db;
    private Logger $logger;
    private string $provider;
    private string $secret;
    private ?DeliveryService $deliveryService = null;
    private ?FraudService $fraudService = null;
    private ?NotificationService $notifier = null;

    public function __construct(PDO $db, Logger $logger, string $provider, string $secret)
    {
        $this->db = $db;
        $this->logger = $logger;
        $this->provider = $provider;
        $this->secret = $secret;
    }

    public function withDelivery(DeliveryService $delivery, FraudService $fraud, NotificationService $notifier): self
    {
        $this->deliveryService = $delivery;
        $this->fraudService = $fraud;
        $this->notifier = $notifier;
        return $this;
    }

    public function createRecharge(int $userId, float $amount, string $method = 'cpay'): int
    {
        $stmt = $this->db->prepare(
            'INSERT INTO recharge_requests (user_id, amount, method, status, created_at) VALUES (:user_id, :amount, :method, :status, NOW())'
        );
        $stmt->execute([
            'user_id' => $userId,
            'amount' => $amount,
            'method' => $method,
            'status' => 'pending',
        ]);
        return (int) $this->db->lastInsertId();
    }

    public function validateSignature(array $payload, string $signature): bool
    {
        $computed = hash_hmac('sha256', json_encode($payload), $this->secret);
        return hash_equals($computed, $signature);
    }

    public function markOrderPaid(int $orderId, string $txId, float $amount): array
    {
        if (!$this->fraudService) {
            throw new \RuntimeException('Fraud service not configured');
        }
        if (!$this->fraudService->checkOrderPayment($orderId, $amount, $txId, $_SERVER['REMOTE_ADDR'] ?? '')) {
            $this->notifier?->notifyTelegram("Order {$orderId} flagged fraud");
            return ['error' => 'fraud_flag'];
        }
        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare('SELECT * FROM orders WHERE id = :id FOR UPDATE');
            $stmt->execute(['id' => $orderId]);
            $order = $stmt->fetch();
            if (!$order) {
                throw new \RuntimeException('Order not found');
            }
            if ($order['status'] === 'paid' || $order['status'] === 'completed') {
                $this->db->commit();
                return ['status' => 'ok', 'idempotent' => true];
            }
            $this->db->prepare('UPDATE orders SET status = "paid", payment_method = :pm WHERE id = :id')
                ->execute(['pm' => $this->provider, 'id' => $orderId]);
            $this->db->prepare(
                'INSERT INTO transactions (user_id, type, amount, meta, created_at) VALUES (:user_id, :type, :amount, :meta, NOW())'
            )->execute([
                'user_id' => $order['user_id'],
                'type' => 'purchase',
                'amount' => $amount,
                'meta' => json_encode(['order_id' => $orderId, 'tx_id' => $txId]),
            ]);
            $this->logger->audit($order['user_id'], 'order_paid', 'orders', $orderId, ['tx' => $txId]);
            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
        if ($this->deliveryService) {
            $delivered = $this->deliveryService->deliver($orderId);
            return ['status' => 'ok', 'delivered' => $delivered];
        }
        return ['status' => 'ok'];
    }

    public function markPaid(int $rechargeId, string $txId): void
    {
        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare('SELECT * FROM recharge_requests WHERE id = :id FOR UPDATE');
            $stmt->execute(['id' => $rechargeId]);
            $req = $stmt->fetch();
            if (!$req || $req['status'] === 'paid') {
                $this->db->commit();
                return;
            }
            $this->db->prepare('UPDATE recharge_requests SET status = :status, ref_id = :ref WHERE id = :id')
                ->execute(['status' => 'paid', 'ref' => $txId, 'id' => $rechargeId]);
            $this->db->prepare('UPDATE users SET balance = balance + :amount WHERE id = :user_id')
                ->execute(['amount' => $req['amount'], 'user_id' => $req['user_id']]);
            $this->db->prepare(
                'INSERT INTO transactions (user_id, type, amount, meta, created_at) VALUES (:user_id, :type, :amount, :meta, NOW())'
            )->execute([
                'user_id' => $req['user_id'],
                'type' => 'topup',
                'amount' => $req['amount'],
                'meta' => json_encode(['tx_id' => $txId]),
            ]);
            $this->logger->audit($req['user_id'], 'recharge_paid', 'recharge_requests', $rechargeId, ['tx' => $txId]);
            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}

