<?php

namespace App\services;

use App\core\Logger;
use PDO;

class OrderService
{
    private PDO $db;
    private DiscountService $discountService;
    private Logger $logger;

    public function __construct(PDO $db, DiscountService $discountService, Logger $logger)
    {
        $this->db = $db;
        $this->discountService = $discountService;
        $this->logger = $logger;
    }

    public function createOrder(int $userId, array $cart, ?string $discountCode = null, ?int $affiliateId = null): int
    {
        if (empty($cart)) {
            throw new \InvalidArgumentException('Cart empty');
        }
        $productIds = array_keys($cart);
        $stmt = $this->db->prepare('SELECT id, price FROM products WHERE id IN (' . implode(',', array_fill(0, count($productIds), '?')) . ')');
        $stmt->execute($productIds);
        $products = [];
        foreach ($stmt->fetchAll() as $row) {
            $products[$row['id']] = (float) $row['price'];
        }
        $subtotal = 0;
        foreach ($cart as $pid => $qty) {
            $price = $products[$pid] ?? 0;
            $subtotal += $price * $qty;
        }
        $discount = $this->discountService->apply($discountCode, $subtotal, $userId);
        $total = max(0, $subtotal - $discount['discount']);

        $this->db->beginTransaction();
        try {
            $stmt = $this->db->prepare(
                'INSERT INTO orders (user_id, total, status, payment_method, affiliate_id, created_at, discount_code, discount_amount) 
                 VALUES (:user_id, :total, :status, :payment_method, :affiliate_id, NOW(), :discount_code, :discount_amount)'
            );
            $stmt->execute([
                'user_id' => $userId,
                'total' => $total,
                'status' => 'pending',
                'payment_method' => 'balance',
                'affiliate_id' => $affiliateId,
                'discount_code' => $discount['code'],
                'discount_amount' => $discount['discount'],
            ]);
            $orderId = (int) $this->db->lastInsertId();
            $itemStmt = $this->db->prepare(
                'INSERT INTO order_items (order_id, product_id, unit_price, qty) VALUES (:order_id, :product_id, :unit_price, :qty)'
            );
            foreach ($cart as $pid => $qty) {
                $itemStmt->execute([
                    'order_id' => $orderId,
                    'product_id' => $pid,
                    'unit_price' => $products[$pid] ?? 0,
                    'qty' => $qty,
                ]);
            }
            $this->logger->audit($userId, 'order_created', 'orders', $orderId, ['total' => $total, 'discount' => $discount]);
            $this->db->commit();
            return $orderId;
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}

