<?php

namespace App\services;

use App\core\Encryptor;
use App\core\Logger;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PDO;

class ExcelImporter
{
    private PDO $db;
    private Encryptor $encryptor;
    private Logger $logger;

    public function __construct(PDO $db, Encryptor $encryptor, Logger $logger)
    {
        $this->db = $db;
        $this->encryptor = $encryptor;
        $this->logger = $logger;
    }

    public function import(string $filePath, int $productId, array $mapping): array
    {
        $spreadsheet = IOFactory::load($filePath);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray(null, true, true, true);

        $success = 0;
        $errors = [];
        $rowNum = 0;
        foreach ($rows as $row) {
            $rowNum++;
            if ($rowNum === 1 && $this->looksLikeHeader($row)) {
                continue;
            }
            $payload = $this->mapRow($row, $mapping);
            if (empty($payload)) {
                $errors[] = ['row' => $rowNum, 'error' => 'Empty mapped data'];
                continue;
            }
            try {
                $blob = $this->encryptor->encrypt($payload);
                $stmt = $this->db->prepare(
                    'INSERT INTO product_variants (product_id, sku, encrypted_data, status, meta, created_at) 
                     VALUES (:product_id, :sku, :encrypted_data, :status, :meta, NOW())'
                );
                $stmt->execute([
                    'product_id' => $productId,
                    'sku' => $payload['sku'] ?? uniqid('sku_'),
                    'encrypted_data' => json_encode($blob),
                    'status' => 'available',
                    'meta' => json_encode(['mapping' => $mapping]),
                ]);
                $success++;
            } catch (\Throwable $e) {
                $errors[] = ['row' => $rowNum, 'error' => $e->getMessage()];
            }
        }

        $this->logger->audit(null, 'excel_import', 'product_variants', $productId, [
            'success' => $success,
            'errors' => $errors,
        ]);

        return ['total' => count($rows), 'success' => $success, 'errors' => $errors];
    }

    private function mapRow(array $row, array $mapping): array
    {
        $payload = [];
        foreach ($mapping as $column => $field) {
            if (isset($row[$column])) {
                $payload[$field] = $row[$column];
            }
        }
        return $payload;
    }

    private function looksLikeHeader(array $row): bool
    {
        $nonNumeric = array_filter($row, fn($v) => !is_numeric($v));
        return count($nonNumeric) === count($row);
    }
}

