<?php

namespace App\services;

use App\core\Encryptor;
use App\core\Logger;
use PDO;
use App\services\AffiliateService;
use App\services\NotificationService;

class DeliveryService
{
    private PDO $db;
    private Encryptor $encryptor;
    private Logger $logger;
    private NotificationService $notifier;
    private ?AffiliateService $affiliateService = null;

    public function __construct(PDO $db, Encryptor $encryptor, Logger $logger, NotificationService $notifier, ?AffiliateService $affiliateService = null)
    {
        $this->db = $db;
        $this->encryptor = $encryptor;
        $this->logger = $logger;
        $this->notifier = $notifier;
        $this->affiliateService = $affiliateService;
    }

    public function deliver(int $orderId): array
    {
        $this->db->beginTransaction();
        try {
            $orderStmt = $this->db->prepare('SELECT * FROM orders WHERE id = :id FOR UPDATE');
            $orderStmt->execute(['id' => $orderId]);
            $order = $orderStmt->fetch();
            if (!$order) {
                throw new \RuntimeException('Order not found');
            }
            if (!in_array($order['status'], ['paid', 'completed'], true)) {
                throw new \RuntimeException('Order not paid');
            }

            $items = $this->db->prepare('SELECT * FROM order_items WHERE order_id = :order_id');
            $items->execute(['order_id' => $orderId]);
            $itemRows = $items->fetchAll();

            $deliveries = [];
            foreach ($itemRows as $item) {
                $variants = $this->db->prepare(
                    'SELECT * FROM product_variants WHERE product_id = :pid AND status = "available" LIMIT ' . (int) $item['qty'] . ' FOR UPDATE'
                );
                $variants->execute(['pid' => $item['product_id']]);
                $variantRows = $variants->fetchAll();
                if (count($variantRows) < $item['qty']) {
                    $this->notifier->notifyTelegram("Low stock for product {$item['product_id']} on order {$orderId}");
                    throw new \RuntimeException('Insufficient stock');
                }
                foreach (array_slice($variantRows, 0, $item['qty']) as $variant) {
                    $this->db->prepare('UPDATE product_variants SET status = "sold" WHERE id = :id')
                        ->execute(['id' => $variant['id']]);
                    $this->db->prepare('UPDATE order_items SET variant_id = :vid WHERE id = :id')
                        ->execute(['vid' => $variant['id'], 'id' => $item['id']]);
                    $data = $this->encryptor->decrypt(json_decode($variant['encrypted_data'], true), (int) $variant['id']);
                    $deliveries[] = [
                        'order_item_id' => $item['id'],
                        'variant_id' => $variant['id'],
                        'data' => $data,
                    ];
                }
            }
            $this->db->prepare('UPDATE orders SET status = "completed", completed_at = NOW() WHERE id = :id')
                ->execute(['id' => $orderId]);

            if ($this->affiliateService && $order['affiliate_id']) {
                $bonus = round(((float) $order['total']) * 0.05, 2);
                $this->affiliateService->recordBonus((int) $order['affiliate_id'], $orderId, $bonus);
            }
            $this->logger->audit($order['user_id'], 'order_delivered', 'orders', $orderId, ['deliveries' => count($deliveries)]);
            $this->db->commit();
            return $deliveries;
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}

