<?php

namespace App\core;

class Encryptor
{
    private string $masterKey;

    public function __construct(string $masterKey)
    {
        if (empty($masterKey)) {
            throw new \InvalidArgumentException('Master key required');
        }
        $this->masterKey = $masterKey;
    }

    public function encrypt(array $data, ?int $variantId = null): array
    {
        $plaintext = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $iv = random_bytes(16);
        $key = $this->deriveKey($variantId);
        $ciphertext = openssl_encrypt($plaintext, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        return [
            'ciphertext' => base64_encode($ciphertext),
            'iv' => base64_encode($iv),
            'tag' => base64_encode($tag),
            'algo' => 'aes-256-gcm',
        ];
    }

    public function decrypt(array $blob, ?int $variantId = null): array
    {
        $key = $this->deriveKey($variantId);
        $iv = base64_decode($blob['iv'] ?? '');
        $ciphertext = base64_decode($blob['ciphertext'] ?? '');
        $tag = base64_decode($blob['tag'] ?? '');
        $plaintext = openssl_decrypt($ciphertext, $blob['algo'] ?? 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $iv, $tag);
        if ($plaintext === false) {
            throw new \RuntimeException('Decryption failed or tampered data');
        }
        return json_decode($plaintext, true);
    }

    private function deriveKey(?int $variantId = null): string
    {
        $salt = $variantId ? pack('N', $variantId) : random_bytes(8);
        return hash_hmac('sha256', $salt, $this->masterKey, true);
    }
}

