<?php

namespace App\core;

use PDO;

class Auth
{
    private PDO $db;
    private int $maxFailed;
    private int $lockMinutes;

    public function __construct(PDO $db, int $maxFailed = 5, int $lockMinutes = 15)
    {
        session_start();
        $this->db = $db;
        $this->maxFailed = $maxFailed;
        $this->lockMinutes = $lockMinutes;
    }

    public function attempt(string $email, string $password): bool
    {
        $stmt = $this->db->prepare('SELECT id, password_hash, role, pin_hash, failed_attempts, locked_until FROM users WHERE email = :email');
        $stmt->execute(['email' => $email]);
        $user = $stmt->fetch();
        if (!$user) {
            return false;
        }
        if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
            return false;
        }
        if (!password_verify($password, $user['password_hash'])) {
            $this->incrementFailure((int) $user['id'], (int) $user['failed_attempts']);
            return false;
        }
        $this->resetFailure((int) $user['id']);
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['role'] = $user['role'];
        return true;
    }

    private function incrementFailure(int $userId, int $current): void
    {
        $current++;
        $lockedUntil = null;
        if ($current >= $this->maxFailed) {
            $lockedUntil = (new \DateTime("+{$this->lockMinutes} minutes"))->format('Y-m-d H:i:s');
        }
        $this->db->prepare('UPDATE users SET failed_attempts = :fa, locked_until = :lu WHERE id = :id')
            ->execute(['fa' => $current, 'lu' => $lockedUntil, 'id' => $userId]);
    }

    private function resetFailure(int $userId): void
    {
        $this->db->prepare('UPDATE users SET failed_attempts = 0, locked_until = NULL WHERE id = :id')
            ->execute(['id' => $userId]);
    }

    public function user(): ?array
    {
        if (!isset($_SESSION['user_id'])) {
            return null;
        }
        $stmt = $this->db->prepare('SELECT id, email, role, balance, pin_hash FROM users WHERE id = :id');
        $stmt->execute(['id' => $_SESSION['user_id']]);
        return $stmt->fetch() ?: null;
    }

    public function checkAdminPin(string $pin): bool
    {
        $user = $this->user();
        if (!$user || !$user['pin_hash']) {
            return false;
        }
        return password_verify($pin, $user['pin_hash']);
    }

    public function requireRole(string $role): void
    {
        if (!isset($_SESSION['role']) || $_SESSION['role'] !== $role) {
            http_response_code(403);
            exit('Forbidden');
        }
    }

    public function requireUser(): void
    {
        if (!isset($_SESSION['user_id'])) {
            http_response_code(401);
            exit('Unauthorized');
        }
    }
}

