<?php

namespace App\controllers;

use App\core\Auth;
use App\core\Controller;
use App\core\Encryptor;
use App\services\CartService;
use App\services\DeliveryService;
use App\services\OrderService;
use App\services\PaymentService;
use PDO;

class OrderController extends Controller
{
    private PDO $db;
    private Auth $auth;
    private CartService $cart;
    private OrderService $orderService;
    private PaymentService $paymentService;
    private DeliveryService $deliveryService;
    private Encryptor $encryptor;

    public function __construct(
        PDO $db,
        Auth $auth,
        CartService $cart,
        OrderService $orderService,
        PaymentService $paymentService,
        DeliveryService $deliveryService,
        Encryptor $encryptor
    ) {
        $this->db = $db;
        $this->auth = $auth;
        $this->cart = $cart;
        $this->orderService = $orderService;
        $this->paymentService = $paymentService;
        $this->deliveryService = $deliveryService;
        $this->encryptor = $encryptor;
    }

    public function addToCart(): void
    {
        $payload = json_decode(file_get_contents('php://input'), true) ?? [];
        $this->cart->add((int) ($payload['product_id'] ?? 0), (int) ($payload['qty'] ?? 1));
        $this->json(['status' => 'ok', 'cart' => $this->cart->items()]);
    }

    public function viewCart(): void
    {
        $items = $this->cart->items();
        $this->json(['cart' => $items]);
    }

    public function checkout(): void
    {
        $this->auth->requireUser();
        $payload = json_decode(file_get_contents('php://input'), true) ?? [];
        $discountCode = $payload['discount_code'] ?? null;
        $affiliateId = $payload['affiliate_id'] ?? null;
        try {
            $orderId = $this->orderService->createOrder($this->auth->user()['id'], $this->cart->items(), $discountCode, $affiliateId);
            $this->cart->clear();
            $this->json(['status' => 'ok', 'order_id' => $orderId]);
        } catch (\Throwable $e) {
            $this->json(['error' => $e->getMessage()], 400);
        }
    }

    public function pay(): void
    {
        $this->auth->requireUser();
        $payload = json_decode(file_get_contents('php://input'), true) ?? [];
        $orderId = (int) ($payload['order_id'] ?? 0);
        $method = $payload['method'] ?? 'balance';

        if ($method === 'balance') {
            $user = $this->auth->user();
            $pin = $payload['pin'] ?? '';
            if (!$this->auth->checkAdminPin($pin)) {
                $this->json(['error' => 'PIN required'], 403);
                return;
            }
            $stmt = $this->db->prepare('SELECT total, status FROM orders WHERE id = :id AND user_id = :uid');
            $stmt->execute(['id' => $orderId, 'uid' => $user['id']]);
            $order = $stmt->fetch();
            if (!$order) {
                $this->json(['error' => 'Order not found'], 404);
                return;
            }
            if ($order['status'] !== 'pending') {
                $this->json(['error' => 'Order not pending'], 400);
                return;
            }
            if ($user['balance'] < $order['total']) {
                $this->json(['error' => 'Insufficient balance'], 400);
                return;
            }
            $this->db->beginTransaction();
            try {
                $this->db->prepare('UPDATE users SET balance = balance - :amt WHERE id = :id')
                    ->execute(['amt' => $order['total'], 'id' => $user['id']]);
                $this->db->prepare('UPDATE orders SET status = "paid", payment_method = "balance" WHERE id = :id')
                    ->execute(['id' => $orderId]);
                $this->db->prepare(
                    'INSERT INTO transactions (user_id, type, amount, meta, created_at) VALUES (:user_id, :type, :amount, :meta, NOW())'
                )->execute([
                    'user_id' => $user['id'],
                    'type' => 'purchase',
                    'amount' => $order['total'],
                    'meta' => json_encode(['order_id' => $orderId, 'method' => 'balance']),
                ]);
                $this->db->commit();
            } catch (\Throwable $e) {
                $this->db->rollBack();
                throw $e;
            }
            $delivered = $this->deliveryService->deliver($orderId);
            $this->json(['status' => 'ok', 'delivered' => $delivered]);
            return;
        }

        // gateway path: return stub payment link with reference
        $this->json([
            'status' => 'pending_gateway',
            'payment' => [
                'provider' => $method,
                'reference_id' => $orderId,
                'reference_type' => 'order',
            ],
        ]);
    }

    public function listOrders(): void
    {
        $this->auth->requireUser();
        $stmt = $this->db->prepare('SELECT * FROM orders WHERE user_id = :uid ORDER BY created_at DESC');
        $stmt->execute(['uid' => $this->auth->user()['id']]);
        $this->json(['orders' => $stmt->fetchAll()]);
    }

    public function showOrder(array $params): void
    {
        $this->auth->requireUser();
        $stmt = $this->db->prepare('SELECT * FROM orders WHERE id = :id AND user_id = :uid');
        $stmt->execute(['id' => $params['id'] ?? 0, 'uid' => $this->auth->user()['id']]);
        $order = $stmt->fetch();
        if (!$order) {
            $this->json(['error' => 'Not found'], 404);
            return;
        }
        $itemsStmt = $this->db->prepare('SELECT * FROM order_items WHERE order_id = :order_id');
        $itemsStmt->execute(['order_id' => $order['id']]);
        $items = $itemsStmt->fetchAll();
        $delivered = [];
        foreach ($items as $item) {
            if ($item['variant_id']) {
                $v = $this->db->prepare('SELECT encrypted_data FROM product_variants WHERE id = :id');
                $v->execute(['id' => $item['variant_id']]);
                $variant = $v->fetch();
                if ($variant) {
                    $delivered[] = [
                        'order_item_id' => $item['id'],
                        'variant_id' => $item['variant_id'],
                        'data' => $this->encryptor->decrypt(json_decode($variant['encrypted_data'], true), (int) $item['variant_id']),
                    ];
                }
            }
        }
        $this->json(['order' => $order, 'items' => $items, 'deliveries' => $delivered]);
    }
}

