<?php

namespace App\controllers;

use App\core\Auth;
use App\core\Controller;
use App\core\Encryptor;
use App\core\Logger;
use App\services\ExcelImporter;
use App\services\NotificationService;
use PDO;

class AdminController extends Controller
{
    private PDO $db;
    private Auth $auth;
    private Encryptor $encryptor;
    private Logger $logger;
    private ExcelImporter $importer;
    private NotificationService $notifier;

    public function __construct(PDO $db, Auth $auth, Encryptor $encryptor, Logger $logger, ExcelImporter $importer, NotificationService $notifier)
    {
        $this->db = $db;
        $this->auth = $auth;
        $this->encryptor = $encryptor;
        $this->logger = $logger;
        $this->importer = $importer;
        $this->notifier = $notifier;
    }

    public function dashboard(): void
    {
        $this->auth->requireRole('admin');
        $stats = $this->db->query('SELECT COUNT(*) as users, SUM(balance) as total_balance FROM users')->fetch();
        $this->json(['stats' => $stats]);
    }

    public function decryptVariant(array $params): void
    {
        $this->auth->requireRole('admin');
        $payload = json_decode(file_get_contents('php://input'), true);
        if (!$this->auth->checkAdminPin($payload['pin'] ?? '')) {
            $this->json(['error' => 'Invalid PIN'], 403);
            return;
        }
        $stmt = $this->db->prepare('SELECT encrypted_data FROM product_variants WHERE id = :id');
        $stmt->execute(['id' => $params['id']]);
        $variant = $stmt->fetch();
        if (!$variant) {
            $this->json(['error' => 'Not found'], 404);
            return;
        }
        $data = $this->encryptor->decrypt(json_decode($variant['encrypted_data'], true), (int) $params['id']);
        $this->logger->audit($this->auth->user()['id'] ?? null, 'decrypt_variant', 'product_variants', (int) $params['id'], []);
        $this->json(['data' => $data]);
    }

    public function orders(): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->query('SELECT * FROM orders ORDER BY created_at DESC LIMIT 200');
        $this->json(['orders' => $stmt->fetchAll()]);
    }

    public function approveOrder(array $params): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->prepare('UPDATE orders SET status = "completed", completed_at = NOW() WHERE id = :id');
        $stmt->execute(['id' => $params['id'] ?? 0]);
        $this->logger->audit($this->auth->user()['id'] ?? null, 'order_approve', 'orders', (int) $params['id'], []);
        $this->json(['status' => 'ok']);
    }

    public function cancelOrder(array $params): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->prepare('UPDATE orders SET status = "cancelled" WHERE id = :id');
        $stmt->execute(['id' => $params['id'] ?? 0]);
        $this->logger->audit($this->auth->user()['id'] ?? null, 'order_cancel', 'orders', (int) $params['id'], []);
        $this->json(['status' => 'ok']);
    }

    public function uploadVariants(array $params): void
    {
        $this->auth->requireRole('admin');
        if (!isset($_FILES['file'])) {
            $this->json(['error' => 'File required'], 400);
            return;
        }
        $mapping = json_decode($_POST['mapping'] ?? '[]', true) ?: [];
        $productId = (int) ($params['id'] ?? 0);
        $result = $this->importer->import($_FILES['file']['tmp_name'], $productId, $mapping);
        $this->json($result);
    }

    public function transactions(): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->query('SELECT * FROM transactions ORDER BY created_at DESC LIMIT 200');
        $this->json(['transactions' => $stmt->fetchAll()]);
    }

    public function adjustBalance(array $params): void
    {
        $this->auth->requireRole('admin');
        $payload = json_decode(file_get_contents('php://input'), true) ?? [];
        if (!$this->auth->checkAdminPin($payload['pin'] ?? '')) {
            $this->json(['error' => 'Invalid PIN'], 403);
            return;
        }
        $delta = (float) ($payload['delta'] ?? 0);
        $this->db->prepare('UPDATE users SET balance = balance + :delta WHERE id = :id')
            ->execute(['delta' => $delta, 'id' => $params['id'] ?? 0]);
        $this->logger->audit($this->auth->user()['id'] ?? null, 'adjust_balance', 'users', (int) ($params['id'] ?? 0), ['delta' => $delta]);
        $this->json(['status' => 'ok']);
    }

    public function settings(): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->query('SELECT `key`, value FROM system_settings');
        $this->json(['settings' => $stmt->fetchAll()]);
    }

    public function updateSettings(): void
    {
        $this->auth->requireRole('admin');
        $payload = json_decode(file_get_contents('php://input'), true) ?? [];
        foreach ($payload as $key => $value) {
            $stmt = $this->db->prepare('REPLACE INTO system_settings (`key`, value) VALUES (:key, :value)');
            $stmt->execute(['key' => $key, 'value' => json_encode($value)]);
        }
        $this->json(['status' => 'ok']);
    }

    public function logs(): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->query('SELECT * FROM logs_audit ORDER BY created_at DESC LIMIT 200');
        $this->json(['logs' => $stmt->fetchAll()]);
    }

    public function users(): void
    {
        $this->auth->requireRole('admin');
        $stmt = $this->db->query('SELECT id, email, role, balance, created_at FROM users ORDER BY created_at DESC LIMIT 200');
        $this->json(['users' => $stmt->fetchAll()]);
    }
}

